<?php

namespace App\Http\Controllers;

use App\Exports\PartyExport;
use App\Exports\TransactionsExport;
use App\Models\Parties;
use App\Models\Reports;
use App\Models\Transaction;
use App\Models\User;
use Dompdf\Dompdf;
use Dompdf\Options;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use Maatwebsite\Excel\Facades\Excel;

class ReportsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $users = User::all();
        $query = Transaction::query();

        if ($request->filled('user')) {
            $query->where(function ($q) use ($request) {
                $q->where('sender', $request->user)
                ->orWhere('receiver', $request->user);
            });
        }

        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->from_date);
        }

        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->to_date);
        }

        $reports = $query->with(['senderUser', 'receiverUser'])->get();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'reports' => $reports]);
        }

        return view('reports.index', compact('reports', 'users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request)
    {
        $ids = explode(',', $request->query('ids'));
        $transactions = Transaction::with(['senderUser', 'receiverUser', 'party'])
            ->whereIn('id', $ids)
            ->get();

        if ($transactions->isEmpty()) {
            return response()->json(['error' => 'No transactions found'], 404);
        }

        $html = View::make('reports.pdf', compact('transactions'))->render();

        $options = new Options();
        $options->set('defaultFont', 'Helvetica');
        $dompdf = new Dompdf($options);
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        $filename = 'Reports_' . now()->format('YmdHis') . '.pdf';

        return $dompdf->stream($filename, ['Attachment' => false]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }


    public function exportExcel(Request $request)
    {
        $ids = explode(',', $request->query('ids'));

        if (empty($ids)) {
            return back()->with('error', 'No transactions selected.');
        }

        return Excel::download(new TransactionsExport($ids), 'transactions_report.xlsx');
    }

    public function exportExcelAPI(Request $request)
    {
        $ids = $request->input('ids');

        if (empty($ids) || !is_array($ids)) {
            return response()->json(['error' => 'No transactions selected.'], 400);
        }

        return Excel::download(new TransactionsExport($ids), 'transactions_report.xlsx');
    }

    public function partyexportExcel(Request $request)
    {
        $ids = explode(',', $request->query('ids'));

        if (empty($ids)) {
            return back()->with('error', 'No transactions selected.');
        }

        return Excel::download(new PartyExport($ids), 'party_transactions_report.xlsx');
    }

    public function partyexportExcelAPI(Request $request)
    {
        $ids = $request->input('ids');

        if (empty($ids) || !is_array($ids)) {
            return response()->json(['error' => 'No transactions selected.'], 400);
        }

        return Excel::download(new PartyExport($ids), 'transactions_report.xlsx');
    }

    public function partyindex(Request $request)
    {
        $parties = Parties::all();
        $query = Transaction::query();

        if ($request->filled('party')) {
            $query->where(function ($q) use ($request) {
                $q->where('sender', $request->user)
                ->orWhere('receiver', $request->user);
            });
        }

        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->from_date);
        }

        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->to_date);
        }

        $reports = $query->with(['party'])->get();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'reports' => $reports]);
        }

        return view('reports.party', compact('reports', 'parties'));
    }

    public function partyPdf(Request $request)
    {
        $ids = explode(',', $request->query('ids'));
        $transactions = Transaction::with(['party'])
            ->whereIn('id', $ids)
            ->get();

        if ($transactions->isEmpty()) {
            return response()->json(['error' => 'No transactions found'], 404);
        }

        $html = View::make('reports.partypdf', compact('transactions'))->render();

        $options = new Options();
        $options->set('defaultFont', 'Helvetica');
        $dompdf = new Dompdf($options);
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        $filename = 'Party_Reports_' . now()->format('YmdHis') . '.pdf';

        return $dompdf->stream($filename, ['Attachment' => false]);
    }

    public function exportPDF(Request $request)
    {
        $ids = $request->input('ids', []);

        if (empty($ids)) {
            return response()->json(['error' => 'No records selected'], 400);
        }

        $transactions = Transaction::with(['senderUser', 'receiverUser', 'party'])
            ->whereIn('id', $ids)
            ->get();

        if ($transactions->isEmpty()) {
            return response()->json(['error' => 'No transactions found'], 404);
        }

        $html = View::make('reports.pdf', compact('transactions'))->render();

        $options = new Options();
        $options->set('defaultFont', 'Helvetica');

        $dompdf = new Dompdf($options);
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        return response($dompdf->output(), 200)
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="User_Report.pdf"');
    }

    public function exportpartyPDF(Request $request)
    {
        $ids = $request->input('ids', []);

        if (empty($ids)) {
            return response()->json(['error' => 'No records selected'], 400);
        }

        $transactions = Transaction::with(['party'])
            ->whereIn('id', $ids)
            ->get();

        if ($transactions->isEmpty()) {
            return response()->json(['error' => 'No transactions found'], 404);
        }

        $html = View::make('reports.partypdf', compact('transactions'))->render();

        $options = new Options();
        $options->set('defaultFont', 'Helvetica');

        $dompdf = new Dompdf($options);
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        return response($dompdf->output(), 200)
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="Party_Report.pdf"');
    }
}
