<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Commission;
use App\Models\MyCommission;
use App\Models\Parties;
use App\Models\PartiesCommission;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;

class TransactionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Transaction::with(['senderUser', 'receiverUser', 'party']);

        if ($request->filled('party')) {
            $query->where('parties_id', $request->party);
        }

        if ($request->filled('date')) {
            $query->whereDate('date', '=', $request->date);
        }

        if ($request->filled('sender_city')) {
            $query->where('sender_city', 'LIKE', '%' . $request->sender_city . '%');
        }

        if ($request->filled('receiver_city')) {
            $query->where('receiver_city', 'LIKE', '%' . $request->receiver_city . '%');
        }

        $transactions = $query->get();

        $parties = Parties::all();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'transactions' => $transactions]);
        }

        return view('transactions.index', compact('transactions', 'parties'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $users = User::where('user_type', '!=', 'Admin')->get();
        $parties = Parties::all();

        return view('transactions.create', compact('users', 'parties'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'sender' => 'required|integer',
            'receiver' => 'required|integer',
            'party'  => 'required|integer',
            'amount' => 'required|numeric',
            'sender_city' => 'required|string',
            'receiver_city' => 'required|string',
            'date' => 'required|date_format:Y-m-d',
            'commission_paid_by' => 'required|string',
        ]);

        $amount = $request->amount;
        if ($amount < 100000) {
            $commissionAmount  = 100;
        } elseif ($amount >= 100000 && $amount <= 500000) {
            $commissionAmount  = $amount * 0.01;
        } else {
            $commissionAmount  = $amount * 0.02;
        }

        $transaction = new Transaction();
        $transaction->sender = $request->input('sender');
        $transaction->receiver = $request->input('receiver');
        $transaction->parties_id = $request->input('party');
        $transaction->amount = $amount;
        $transaction->sender_city = $request->input('sender_city');
        $transaction->receiver_city = $request->input('receiver_city');
        $transaction->date = $request->input('date');
        $transaction->commission_paid_by = $request->input('commission_paid_by');
        // echo '<pre>';print_r($transaction);exit;
        $transaction->save();

        $commission = new Commission();
        $commission->sender = $request->input('sender');
        $commission->receiver = $request->input('receiver');
        $commission->parties_id = $request->input('party');
        $commission->date = $request->input('date');
        $commission->amount = $amount;
        $commission->sender_city = $request->input('sender_city');
        $commission->receiver_city = $request->input('receiver_city');
        $commission->total_commission = $commissionAmount ;
        // echo '<pre>';print_r($commission);exit;
        $commission->save();

        $senderAccount = Account::where('user_id', $request->input('sender'))->first();
        $receiverAccount = Account::where('user_id', $request->input('receiver'))->first();
        $senderUser = User::find($request->input('sender'));
        $receiverUser = User::find($request->input('receiver'));

        if (!$senderAccount) {
            $senderAccount = new Account();
            $senderAccount->user_id = $senderUser->id;
            $senderAccount->email = $senderUser->email;
            $senderAccount->phone_number = $senderUser->phone_number;
            $senderAccount->balance = 0.00;
            $senderAccount->save();
        }

        if (!$receiverAccount) {
            $receiverAccount = new Account();
            $receiverAccount->user_id = $receiverUser->id;
            $receiverAccount->email = $receiverUser->email;
            $receiverAccount->phone_number = $receiverUser->phone_number;
            $receiverAccount->balance = 0.00;
            $receiverAccount->save();
        }

        if ($request->input('commission_paid_by') === 'Sender') {
            // Deduct commission from sender
            $senderAccount->balance -= ($amount + $commissionAmount);
            $receiverAccount->balance += $amount;
        } else {
            // Deduct commission from receiver
            $senderAccount->balance -= $amount;
            $receiverAccount->balance += ($amount - $commissionAmount);
        }

        $senderAccount->save();
        $receiverAccount->save();

        $party = Parties::find($request->input('party'));
        // For example, if party->commission_percentage is 50, then:
        // Party gets 50% of the total commission and "my commission" gets the remaining 50%.
        $partyCommissionPercentage = $party ? $party->parties_commission : 0;
        $partyCommissionShare = $commissionAmount * ($partyCommissionPercentage / 100);
        $myCommissionShare = $commissionAmount - $partyCommissionShare;

        // Create an entry for Party Commission.
        $partyCommission = new PartiesCommission();
        $partyCommission->sender = $request->input('sender');
        $partyCommission->receiver = $request->input('receiver');
        $partyCommission->parties_id = $request->input('party');
        $partyCommission->date = $request->input('date');
        $partyCommission->amount = $amount;
        $partyCommission->party_commission = $partyCommissionShare;
        $partyCommission->sender_city = $request->input('sender_city');
        $partyCommission->receiver_city = $request->input('receiver_city');
        $partyCommission->save();

        // Create an entry for My Commission.
        $myCommission = new MyCommission();
        $myCommission->sender = $request->input('sender');
        $myCommission->receiver = $request->input('receiver');
        $myCommission->parties_id = $request->input('party');
        $myCommission->date = $request->input('date');
        $myCommission->amount = $amount;
        $myCommission->my_commission = $myCommissionShare;
        $myCommission->sender_city = $request->input('sender_city');
        $myCommission->receiver_city = $request->input('receiver_city');
        $myCommission->save();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'message' => 'Transaction Created Successfully', 'transaction' => $transaction]);
        }

        return redirect()->route('transactions.index')->with('success', 'Transaction Created Successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id, Request $request)
    {
        $transaction = Transaction::findOrFail($id);
        $transaction->delete();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'message' => 'Transaction Deleted Successfully', 'transaction' => $transaction]);
        }

        return redirect()->route('transactions.index')->with('success','Transaction Deleted successfully');
    }

    public function bulkDestroy(Request $request)
    {
        $ids = json_decode($request->ids, true);
        // echo '<pre>';print_r($ids);exit;
        if (!empty($ids)) {
            Transaction::whereIn('id', $ids)->delete();
            return redirect()->route('transactions.index')->with('success', 'Selected transactions deleted successfully.');
        }
        return back()->with('error', 'No records selected.');
    }
}
