<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\DataTables\UsersDataTable;
use App\Models\User;
use App\Helpers\AuthHelper;
use Spatie\Permission\Models\Role;
use App\Http\Requests\UserRequest;
use App\Models\Account;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(UsersDataTable $dataTable, Request $request)
    {
        $users = User::where('user_type', '!=', 'Admin')->get();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'users' => $users]);
        }

        return view('users.list', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('users.form');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = new User();
        $user->first_name = $request->input('first_name');
        $user->last_name = $request->input('last_name');
        $user->email = $request->input('email');
        $user->phone_number = $request->input('phone_number');
        // echo '<pre>';print_r($user);exit;
        $user->save();

        $account = new Account();
        $account->user_id = $user->id;
        $account->email = $user->email;
        $account->phone_number = $user->phone_number;
        $account->balance = 0.00;
        // echo '<pre>';print_r($account);exit;
        $account->save();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'message' => 'User and Account Created Successfully', 'user' => $user]);
        }

        return redirect()->route('users.index')->with('success', 'User and Account Created Successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id, Request $request)
    {
        $data = User::with('userProfile','roles')->findOrFail($id);

        $profileImage = getSingleMedia($data, 'profile_image');

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'user' => $data, 'profile_image' => $profileImage]);
        }

        return view('users.profile', compact('data', 'profileImage'));
    }

    public function userShow($id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['error' => 'User not found'], 404);
        }
        return response()->json($user);
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id, Request $request)
    {
        $user = User::findOrFail($id);

        // echo '<pre>';print_r($user);exit;
        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'user' => $user]);
        }

        return view('users.form', compact('user', 'id'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        // dd($request->all());
        $user = User::findOrFail($id);
        $user->first_name = $request->input('first_name');
        $user->last_name = $request->input('last_name');
        $user->email = $request->input('email');
        $user->phone_number = $request->input('phone_number');
        $user->users_commission = $request->input('users_commission');
        $user->save();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'message' => 'User Updated Successfully', 'user' => $user]);
        }

        return redirect()->route('users.index')->with('success', 'User Updated successfully');
    }

    public function userUpdate(Request $request, $id)
    {
        try {
            $user = User::find($id);
            if (!$user) {
                return response()->json(['status' => false, 'message' => 'User not found'], 404);
            }

            // Validate input data
            $validated = $request->validate([
                'first_name' => 'required|string|max:255',
                'last_name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email,' . $id,
                'phone_number' => 'required|string|max:15',
            ]);

            // Update user details
            $user->update($validated);

            return response()->json([
                'status' => true,
                'message' => 'User updated successfully',
                'user' => $user
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error updating user',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id, Request $request)
    {
        $user = User::findOrFail($id);
        $user->delete();

        if ($request->wantsJson()) {
            return response()->json(['status' => true, 'message' => 'User Deleted Successfully', 'user' => $user]);
        }

        return redirect()->route('users.index')->with('success','User Deleted successfully');
    }
}
